function [pyknogram, filtered_signal, filterbank, centerFrequency] = PyknoGram (signal, samplerate, nfilters,ntaps , bandwidth, lowF,hiF, scale)
	% [pyknogram filtered_signal filterbank centerFrequency] = PyknoGram (signal, samplerate, nFilters,nTaps , bandwidth, lowF,hiF, scale)
	%
	%
	% pykfec - pyknogram frequency estimated coefficients toolbox for Matlab/Octave
	% Copyright (C) <2008>  <Marco.Grimaldi@gmail.com>
	%
	%
	% Calculates the _pyknogram_ of an input signal using a bank of Gabor filters [1,2]
	% The algoritm makes use of the Hilbert Transform as opposed to the DESA approach[2]
	%
	% Input and Outputs are as follows :
	% Input:
	%	signal:			input signal
	%	samplerate:		sample rate of the input signal in Hz
	% 	nfilters:		number of desired filters in the bank 	[default: 40]
	% 	ntaps:			number of taps in each filter		[default: pow2(floor(log2(0.03*samplerate)))]
	% 	bandwidth:		bandwidth of the gabor filters in Hz	[default: 160 Hz]
	%	lowF:			center frequency of the lowest Gabor filter in the filterbank	[default: 200]
	%	hiF:			center frequency of the highest Gabor filter in the filterbank	[default: 3800]
	%	scale			scale the bandwidth of the Gabor filters [default: 1]:
	%					0:	linearly spaced Gabor filterbank with constant bandwidth on the Hz scale
	%					1:	linearly spaced Gabor filterbank with constant bandwidth on the Mel scale
	%
	% Output:
	%	pyknogram:		pyknogram of the input signal
	%	filtered_signal:	matrix containing the band passed signal (row-wise)
	%	filterbank: 		matrix nfilters x ntaps containing the filterbank
	%	centerFrequency:	vector containing the (normalized) center frequency of each filter
	%
	%
	%
	%  [1]  Grimaldi M. and Cummins F. (2008). Speaker Identification Using Instantaneous Frequencies. 
	%	IEEE Transaction Audio on Audio, Speech, and Language Processing, 16 (6):1097-1111.
	%
	%  [2]	Potamianos A. and Maragos P. (1996). 
	%	Speech Formant Frequency and Bandwidth Tracking Using Multiband Energy Demodulation.
	%	J. Acoust. Soc. Amer., 99 (6), pp.3795–3806.
	%
	% See also:
	%	GaborFilterBank, LinearGaborFilterBankMelBandwidth, InstFrequency, ShortTimeFrequencyEstimate, PlotPyknoGram

	% Input handling
	if (nargin == 2)
		nfilters = 40;
		ntaps = pow2(floor(log2(0.03*samplerate)));
		bandwidth = 160;
		lowF = 200;
		hiF = 3800;
		scale = 1;
	end	

	if (nargin<2 || (nargin > 2 && nargin <8) || nargin>8 )
		error('Usage: [pyknogram, filtered_signal, filterbank, centerFrequency] = PyknoGram (signal, samplerate, nfilters,ntaps , bandwidth, lowF,hiF, scale)'); 
	end
	if scale > 1 || scale < 0
		error('scale must be a scalar with value between 0 and 1');
	end
	
	%tic
	% adjust input signal length
	coeff = floor(length(signal)/ntaps);
	signal = signal(1:coeff*ntaps);
	
	% get selected filterbank
	if (scale == 0)
		[filterbank centerFrequency]= GaborFilterBank (nfilters, ntaps, bandwidth, samplerate, lowF, hiF);
	elseif (scale == 1)
		[filterbank centerFrequency]= LinearGaborFilterBankMelBandwidth(nfilters, ntaps, bandwidth, samplerate, lowF, hiF);
	end
	
	filterbank = filterbank';
	
	%filter the signal	
	S = fft(signal, length(signal));
	F = fft(filterbank, length(signal));
	filtered_signal = ifft(F.*(S*ones(1,nfilters)));

	%calclate hilbert transform
	ht = hilbert(real(filtered_signal));

	%Instantaneous frequency and ampltude
	ifreq = InstFrequency(ht);
	iamp = sqrt(real(ht).^2+imag(ht).^2);

	pyknogram_length = floor(2*length(filtered_signal)/ntaps)-2;
	pyknogram = zeros(pyknogram_length, nfilters);
	for i = 1:nfilters
		pyknogram(:,i) = ShortTimeFrequencyEstimate(iamp(:,i), ifreq(:,i),ntaps);
	end
	%toc	
end
